﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Collections;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Security.Principal;
using System.Drawing.Printing;
using System.Globalization;
using System.Xml.Serialization;

namespace GLERP
{// Copyright 2021 by Dr Randall J Dyck
 
    // GLERP (General Ledger for Enterprise Resource Planning system)

    // G/L System is described in the book Computerized Financial Systems http://www.amazon.com/dp/150291056X
    // which discusses the data structures and processing for general ledger, accounts receivable, accounts payable,
    // fixed assets, inventory, payroll, sales, and ordering systems.
    // Author profile and other books can be seen at http://amazon.com/author/randalldyck
    // This demo file can be downloaded from http://rdyck.mynucleus.ca

    // Compile this code with C# Express (available as a download from Microsoft)
    // (or Visual Studio Community)
    // Double click on the .SLN file in the first directory after unzipping this source
    //   or do a file open for a project solution and select the .SLN file.
    // Press the Start button or choose the Debug menu and the start option to create the program.
    // The GLERP.EXE program that is created will automatically make its own database when it is used.
    // The program is about 70K in size and its database is less than 100K in size.
    // Once you have located the program, then you can copy it to other directories to make more companies.
    
    // To start using the system:
    // 1. Key in a company name
    // 2. Key in categories for accounts eg. Assets, Current, Sales, ...
    // 3. Key in accounts (types are B for balance sheet and I for income statement)
    // 4. Fill in the retained earnings account by the company name (the drop down may be used to see accounts)
    // 5. Define financial periods
    // 6. Enter a journal # to start a new batch of accounts to put in debit and credit amounts for activities.
    //    (valid types are N for normal batches, A for accruals that auto-reverse, and R for recurring batches)
    // 7. Process your journal batches (you will be told which journals balance and can be posted)
    // 8. View and print your financial reports (and remember to close the year in the last financial period)
    //          (note that the year may be closed many times as it just recalculates the next year open balances)

    // The source code is distributed here so that everyone can build a virus free G/L program customized to their needs
    // and also so that technical staff responsible for financial systems can see how the concepts are used.
    // All of the code has been kept in this single file and basic coding is used to help illustrate concepts.

    // This system is intended for educational purposes but small businesses might also find value in it.

    // Technical enhancements that you can make to this basic system would be:
    // A. Break the single database file into 3 files to speed up the system for many transactions.
    //    The current system saves data whenever anything changes.
    //    By having a file to store the basic information (categories, accounts, periods, and controls),
    //    another for the journal batch header and details, and a third for the summary and detail data
    //    then the system would only have to save the larger files when their data is changed.
    // B. Export and import options could be added to put/retrieve the class data into .CSV files
    //    for easy transport to other programs (like Microsoft Excel) and for bulk inserts and modifications.
    // C. A multi-user system can be created by saving the class data into a real database
    //    by using Datasets on the grids or by using SQL in interface lists (ILIST) instead of SortedLists.

    public partial class Form1 : Form
    {
        string date_format = "M/d/yyyy", date_format2 = "M-d-yyyy";
        string current_user = "", edit_type = "";
        //public static database db = new database();
        public static databaseglerp db = new databaseglerp();
        bool loaded = false, process=true, foundperiod=false;
        double amount = 0;
        int current_period = 0, current_year = 0;
        journal_header header = new journal_header();
        int last_journal = 0, highjournal=0;
        int menurow = 0, menucol = 0;
        ContextMenuStrip cs = new ContextMenuStrip();
        //report control variables (defined here since print method is re-entered multiple times as report pages are printed)
        //   A class could be used here and in other places but then the code would become fragmented instead of being
        //   kept all in one file for study purposes.  Similarly, all methods are defined in this single file.
        int print_year = 0, print_period = 0, print_retained_account = 0, start = 0, lines_used = 0, font_size = 12;
        SolidBrush b = new SolidBrush(Color.Black);
        Font f = new Font(new FontFamily("Courier New"), 12, FontStyle.Regular);
        int column = 0, row = 0;  //note: courier gives mono spacing so that exact formatting can be given
        string report_description = "", printkey = "", print_ytd = "", report_type = "", report_data = "", print_amount = "";
        string last_print_trans_account = "", last_major = "", last_minor = "", start_key = "";
        double major_total = 0, minor_total = 0, final_total = 0, tran_total = 0, net_total = 0;
        bool heading_printed = false, print_header=true, fullpage=false;
        int last_print_account = 0;
        DateTime print_date;

        public Form1()
        {
            InitializeComponent();
        }

        private void Form1_Load(object sender, EventArgs e)
        {//Copyright 2021 Dr Randall J Dyck  http://rdyck.mynucleus.ca
            if (File.Exists(@"glerp.xml")) db.load();
            current_user = getuser(); //get user id
            findperiod(out current_year, out current_period); //find current accounting period
            if (Form1.db.controls.ContainsKey(1)) //get control data
            {
                companyname.Text = Form1.db.controls[1].company;
                reacct.Text = Form1.db.controls[1].reacct.ToString();
            }
            loadcat();       //load dropdown list of categories
            loadlist();      //load dropdown list of accounts
            loadjournals();  //load dropdown list of journals
            get_highest_journal();//get highest posted journal number
            companyname.Select(0, 0);
            loaded = true;
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {//Copyright 2021 Dr Randall J Dyck  http://rdyck.mynucleus.ca
            if (edit_type != "") save_grid();
            db.save();
            //databaseglerp db2 = new databaseglerp();
            //db2.save();
        }

        private void linkLabel1_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {//display website describing this system
            System.Diagnostics.Process.Start("http://rdyck.mynucleus.ca/glerp.htm");
        }

        private string getuser()
        {
            string id = WindowsIdentity.GetCurrent().Name.ToString(); //gives domain\user eg. CORP\ED
            string[] idparts = id.Split(Convert.ToChar(@"\")); //only need user id
            if (idparts.Length > 1) return idparts[1];
            return idparts[0];//default
        }

        private void findperiod(out int year_number, out int period_number) //look up current accounting period
        {
            year_number = period_number = 0;
            DateTime today = new DateTime();
            today = DateTime.Now;
            foreach (financial_period data in Form1.db.periods.Values) //array is small so a loop is used
            {
                if (today.CompareTo(data.ending) > 0) continue;
                year_number = data.year;
                period_number = data.period;
                break;
            }
        }

        private void loadlist() //set up dropdown list of accounts
        {
            contextMenuStrip1 = new ContextMenuStrip();
            int size = Form1.db.accounts.Count;
            if (size > 0)
            {
                string[] list = new string[size];
                int x = 0;
                foreach(account data in Form1.db.accounts.Values)
                {//format will auto-expand if account numbers are more than 10 digits
                    list[x] = string.Format("{0,10} {1,-60}", data.account_number, data.description);
                    contextMenuStrip1.Items.Add(list[x]);
                    x++;
                }
                accountlist.DataSource = list;
                contextMenuStrip1.ItemClicked += menu_clicked;
            }
        }
        private void menu_clicked(object sender,ToolStripItemClickedEventArgs e)
        {
            ToolStripItem i = e.ClickedItem;
            string item = i.Text.Trim();
            int pos = item.IndexOf(" ");
            item = item.Substring(0, pos);
            if (menurow >= 0)
            {
                dataGridView1.Rows[menurow].Cells[0].Value = item;
                dataGridView1.EndEdit();
            }
        }
        private void loadcat() //set up dropdown list of categories
        {
            contextMenuStrip2 = new ContextMenuStrip();
            int size = Form1.db.categories.Count;
            if (size > 0)
            {
                foreach (category data in Form1.db.categories.Values)
                {//format will auto-expand if account numbers are more than 10 digits
                    string catdata = string.Format("{0,5} {1,-35}", data.code, data.description);
                    contextMenuStrip2.Items.Add(catdata);
                }
                contextMenuStrip2.ItemClicked += menucat_clicked;
            }
        }
        private void menucat_clicked(object sender, ToolStripItemClickedEventArgs e)
        {
            ToolStripItem i = e.ClickedItem;
            string item = i.Text.Trim();
            int pos = item.IndexOf(" ");
            item = item.Substring(0, pos);
            if (menurow >= 0)
            {
                dataGridView1.Rows[menurow].Cells[menucol].Value = item;
                dataGridView1.EndEdit();
            }
        }
        private void get_highest_journal()
        {//look through posted journals in G/L details to find highest journal #
            highjournal = 0;
            foreach (detail GL in Form1.db.details.Values)
                if (GL.journal > highjournal) highjournal = GL.journal;
        }

        private void dataGridView1_KeyDown(object sender, KeyEventArgs e)
        {
        }

        private void accountlist_SelectedIndexChanged(object sender, EventArgs e)
        {
            if(accountlist.SelectedItem!=null && loaded)
            {
                string account = accountlist.SelectedItem.ToString().Trim();
                int spacing = account.IndexOf(" ");
                if (spacing == -1) spacing = account.Length;
                try
                {
                    reacct.Text = account.Substring(0, (spacing));
                }
                catch (Exception es) { }
            }
        }

        private void dataGridView1_UserAddedRow(object sender, DataGridViewRowEventArgs e)
        {//if new accounts added then refresh drop down list as soon as each one is added
            if (edit_type == "ACT")
            {
                int size = Form1.db.accounts.Count;
                int size2 = dataGridView1.Rows.Count;
                //note that pressing Enter to open a new row will come into this code
                //before any data is added, and so you must wait until another Enter
                //is pressed before you can see the data on the previous row
                if ((size2 - size) > 1)
                {
                    string[] list = new string[(size2-1)];//skip current blank row
                    int x = 0;
                    for (x = 0; x < size2 - 2; x++)//get previous rows of data
                    {
                        int account_number = 0;
                        if(dataGridView1.Rows[x].Cells["account"].Value!=null)
                            Int32.TryParse(dataGridView1.Rows[x].Cells["account"].Value.ToString(), out account_number);
                        string description = " ";
                        if (dataGridView1.Rows[x].Cells["desc"].Value != null)
                            description = dataGridView1.Rows[x].Cells["desc"].Value.ToString();
                        if(account_number!=0)
                            list[x] = string.Format("{0,10} {1,-60}", account_number, description);
                    }
                    loaded = false;
                    accountlist.DataSource = list;
                    loaded = true;
                }
            }
        }
        private void dataGridView1_CellMouseDown(object sender, DataGridViewCellMouseEventArgs e)
        {
            if(e.Button==MouseButtons.Right && edit_type=="DET")
            {
                menurow = e.RowIndex;
                menucol = e.ColumnIndex;
                contextMenuStrip1.Show(dataGridView1, new Point(e.X, e.Y));
            }
            if (e.Button == MouseButtons.Right && edit_type == "ACT")
            {
                menurow = e.RowIndex;
                menucol = e.ColumnIndex;
                //only show search for major and minor codes
                if(menucol==2 || menucol==3)
                    contextMenuStrip2.Show(dataGridView1, new Point(e.X, e.Y));
            }
        }

        private void dataGridView1_EditingControlShowing(object sender, DataGridViewEditingControlShowingEventArgs e)
        {//stop standard copy/paste popup menu from hiding popup search menus
            e.Control.ContextMenuStrip = cs;
        }

        private void next_Click(object sender, EventArgs e)
        {
            if(loaded)
            { 
                ejournal_TextChanged(sender, e);
                edit_type = "";
                show_details(0);
                last_journal = 0;
                int nexthigh = highjournal, journal = 0;
                //see if any pending journals have a higher number
                foreach (journal_header JH in Form1.db.jheaders.Values)
                    if (JH.journal > nexthigh) nexthigh = JH.journal;
                journal = nexthigh + 1;
                ejournal.Text = journal.ToString();
                journal = 0;
            }
        }

        private void loadjournals() //set up dropdown list of journals
        {
            int size = Form1.db.jheaders.Count;
            if (size > 0)
            {
                string[] list = new string[size];
                int x = 0;
                foreach (journal_header data in Form1.db.jheaders.Values)
                {
                    list[x] = data.journal.ToString();
                    if (x == 0 && !loaded) //show first journal when program loaded
                    {
                        ejournal.Text = data.journal.ToString();
                        type.Text = data.type;
                        year.Text = data.year.ToString();
                        period.Text = data.period.ToString();
                        if (data.year == 0) year.Text = current_year.ToString();
                        if (data.period == 0) period.Text = current_period.ToString();
                        source.Text = data.source;
                        description.Text = data.description;
                        header = Form1.db.jheaders[data.journal];
                        last_journal = data.journal;
                    }
                    x++;
                }
                process = false;  //stop dropdown from activating until user manually selects from it
                journal.DataSource = list;
                process = true;
            }
        }

        private void save_grid()
        {
            for (int x = 0; x < dataGridView1.Rows.Count; x++)
            {
                    if (edit_type == "CAT") //update categories
                    {
                        category datac = new category();
                        if (dataGridView1.Rows[x].Cells["code"].Value != null)
                        {
                            datac.code = dataGridView1.Rows[x].Cells["code"].Value.ToString();
                            if (dataGridView1.Rows[x].Cells["desc"].Value == null)
                                dataGridView1.Rows[x].Cells["desc"].Value = " ";
                            datac.description = dataGridView1.Rows[x].Cells["desc"].Value.ToString();
                            Form1.db.categories[datac.code] = datac;
                        }
                    }
                    if (edit_type == "ACT") //update accounts
                    {
                        account dataa = new account();
                        if (dataGridView1.Rows[x].Cells["account"].Value != null)
                        {
                            int account_number = 0;
                            Int32.TryParse(dataGridView1.Rows[x].Cells["account"].Value.ToString(), out account_number);
                            dataa.account_number = account_number;
                            if (dataGridView1.Rows[x].Cells["desc"].Value == null)
                                dataGridView1.Rows[x].Cells["desc"].Value = " ";
                            dataa.description = dataGridView1.Rows[x].Cells["desc"].Value.ToString();
                            if (dataGridView1.Rows[x].Cells["major"].Value == null)
                                dataGridView1.Rows[x].Cells["major"].Value = " ";
                            dataa.major = dataGridView1.Rows[x].Cells["major"].Value.ToString();
                            if (dataGridView1.Rows[x].Cells["minor"].Value == null)
                                dataGridView1.Rows[x].Cells["minor"].Value = " ";
                            dataa.minor = dataGridView1.Rows[x].Cells["minor"].Value.ToString();
                            if (dataGridView1.Rows[x].Cells["stmt"].Value == null)
                                dataGridView1.Rows[x].Cells["stmt"].Value = "B";
                            string statement = dataGridView1.Rows[x].Cells["stmt"].Value.ToString().ToUpper();
                            if (statement != "B" && statement != "I") statement = "B";
                            dataa.statement = statement;
                            Form1.db.accounts[dataa.account_number] = dataa;
                        }
                    }
                    if (edit_type == "PER") //update financial periods
                    {
                        financial_period datap = new financial_period();
                        if (dataGridView1.Rows[x].Cells["year"].Value != null)
                        {
                            int year = 0, period = 0;
                            DateTime ending = new DateTime();
                            ending = DateTime.Now;
                            Int32.TryParse(dataGridView1.Rows[x].Cells["year"].Value.ToString(), out year);
                            Int32.TryParse(dataGridView1.Rows[x].Cells["period"].Value.ToString(), out period);
                            if (dataGridView1.Rows[x].Cells["ending"].Value == null)
                                dataGridView1.Rows[x].Cells["ending"].Value = DateTime.Now.ToShortDateString();
                            DateTime.TryParseExact(dataGridView1.Rows[x].Cells["ending"].Value.ToString(),
                                date_format, CultureInfo.InvariantCulture, DateTimeStyles.None, out ending);
                            if(ending.ToShortDateString()=="0001-01-01")//use alternate date format
                                DateTime.TryParseExact(dataGridView1.Rows[x].Cells["ending"].Value.ToString(),
                                date_format2, CultureInfo.InvariantCulture, DateTimeStyles.None, out ending);
                            string key = string.Format("{0:0000}{1:00}", year, period);
                            datap.year = year;
                            datap.period = period;
                            datap.ending = ending;
                            Form1.db.periods[key] = datap;
                        }
                    findperiod(out current_year, out current_period); //find current accounting period
                }
                    if (edit_type == "BAT") //update journal headers
                    {
                        journal_header datab = new journal_header();
                        int journal = 0, year = 0, period = 0;
                        DateTime tdate = new DateTime();
                        tdate = DateTime.Now;
                        if (dataGridView1.Rows[x].Cells["journal"].Value != null)
                        {
                            Int32.TryParse(dataGridView1.Rows[x].Cells["journal"].Value.ToString(), out journal);
                            Int32.TryParse(dataGridView1.Rows[x].Cells["year"].Value.ToString(), out year);
                            Int32.TryParse(dataGridView1.Rows[x].Cells["period"].Value.ToString(), out period);
                            DateTime.TryParseExact(dataGridView1.Rows[x].Cells["date"].Value.ToString() + " " +
                              dataGridView1.Rows[x].Cells["time"].Value.ToString(),
                              date_format+" h:mm tt", CultureInfo.InvariantCulture, DateTimeStyles.None, out tdate);
                            if(tdate.ToShortDateString()=="0001-01-01")//use alternate date format
                              DateTime.TryParseExact(dataGridView1.Rows[x].Cells["date"].Value.ToString() + " " +
                              dataGridView1.Rows[x].Cells["time"].Value.ToString(),
                              date_format2 + " h:mm tt", CultureInfo.InvariantCulture, DateTimeStyles.None, out tdate);
                            datab.journal = journal;
                            datab.detail_date_time = tdate;
                            datab.year = year;
                            datab.period = period;
                            datab.description = dataGridView1.Rows[x].Cells["desc"].Value.ToString();
                            datab.source = dataGridView1.Rows[x].Cells["source"].Value.ToString();
                            datab.userid = dataGridView1.Rows[x].Cells["user"].Value.ToString();
                            datab.type = dataGridView1.Rows[x].Cells["type"].Value.ToString().ToUpper();
                            Form1.db.jheaders[journal] = datab;
                        }
                    }
                    if (edit_type == "DET") //update journal details
                    {
                        journal_detail datad = new journal_detail();
                        int account_number = 0;
                        double amount = 0;
                        if (dataGridView1.Rows[x].Cells["account"].Value != null && last_journal>0)
                        {
                            Int32.TryParse(dataGridView1.Rows[x].Cells["account"].Value.ToString(), out account_number);
                        if (dataGridView1.Rows[x].Cells["amount"].Value == null) MessageBox.Show("You forgot to press Enter after your last amount");
                        else
                        {
                            double.TryParse(dataGridView1.Rows[x].Cells["amount"].Value.ToString(), out amount);
                            datad.journal = last_journal;
                            datad.account_number = account_number;
                            datad.amount = amount;
                            string key = string.Format("{0,20}{1,20}", datad.journal, datad.account_number);
                            Form1.db.jdetails[key] = datad;
                        }
                        }
                    }
            }
            save_header();
            if (edit_type == "ACT")
            {
                loaded = false;
                loadlist(); //reload dropdown account list
                loaded = true;
            }
            if (edit_type == "CAT")
            {
                loaded = false;
                loadcat(); //reload dropdown category list
                loaded = true;
            }
            edit_type = "";
            db.save();
        }

        private void save_header()
        {
            if (last_journal != 0) //process current journal header
            {
                header = new journal_header();
                int year_number = 0, period_number = 0;
                header.detail_date_time = DateTime.Now;
                header.userid = current_user;
                header.description = description.Text;
                header.source = source.Text;
                if (type.Text != "N" && type.Text != "A" && type.Text != "R")
                    type.Text = "N";
                header.type = type.Text;
                year_number = 0;
                Int32.TryParse(year.Text, out year_number);
                if(year_number==0)
                {
                    year_number = current_year;
                    year.Text = current_year.ToString();
                }
                header.year = year_number;
                period_number = 0;
                Int32.TryParse(period.Text, out period_number);
                if(period_number==0)
                {
                    period_number = current_period;
                    period.Text = current_period.ToString();
                }
                header.period = period_number;
                header.journal = last_journal;
                //ignore journals that have no description and no details
                bool details_found = false;
                if (description.Text != "") details_found = true;
                else
                {
                    foreach (journal_detail record in Form1.db.jdetails.Values)
                        if (record.journal == last_journal)
                        {
                            details_found = true;
                            break;
                        }
                }
                if (details_found) //save journal
                {
                    Form1.db.jheaders[last_journal] = header;
                    header = new journal_header();
                }
            }
        }

        private void dataGridView1_UserDeletingRow(object sender, DataGridViewRowCancelEventArgs e)
        {
            if (edit_type == "CAT") Form1.db.categories.Remove(e.Row.Cells["code"].Value.ToString());
            if (edit_type == "ACT")
            {
                int account_number = 0;
                Int32.TryParse(e.Row.Cells["account"].Value.ToString(), out account_number);
                Form1.db.accounts.Remove(account_number);
            }
            if (edit_type == "BAT")
            {
                int journal = 0;
                Int32.TryParse(e.Row.Cells["journal"].Value.ToString(), out journal);
                Form1.db.jheaders.Remove(journal);
                //gather detail keys to delete
                List<string> keys = new List<string>();
                foreach (journal_detail jd in Form1.db.jdetails.Values)
                {
                    if (jd.journal == journal)
                    {
                        string jkey = string.Format("{0,20}{1,20}", jd.journal, jd.account_number);
                        keys.Add(jkey);
                    }
                }
                //now remove journal details using batch detail keys to delete
                foreach(string jdelete in keys)
                    Form1.db.jdetails.Remove(jdelete);
            }
            if (edit_type == "PER")
            {
                int year = 0, period = 0;
                Int32.TryParse(e.Row.Cells["year"].Value.ToString(), out year);
                Int32.TryParse(e.Row.Cells["period"].Value.ToString(), out period);
                string key = string.Format("{0:0000}{1:00}", year, period);
                Form1.db.periods.Remove(key);
            }
            if (edit_type == "DET")
            {
                int account_number = 0;
                Int32.TryParse(e.Row.Cells["account"].Value.ToString(), out account_number);
                string key = string.Format("{0,20}{1,20}", last_journal, account_number);
                Form1.db.jdetails.Remove(key);
            }
        }

        private void maintaincat_Click(object sender, EventArgs e)
        {
            if (edit_type != "") save_grid();
            edit_type = "CAT";
            dataGridView1.Columns.Clear();
            dataGridView1.Rows.Clear();
            dataGridView1.Columns.Add("code", "Category");
            dataGridView1.Columns.Add("desc", "Description");
            int size = Form1.db.categories.Count;
            if (size > 0)
            {
                foreach (category data in Form1.db.categories.Values)
                {
                    string[] row = { data.code, data.description };
                    dataGridView1.Rows.Add(row);
                }
            }
        }

        private void maintainacct_Click(object sender, EventArgs e)
        {
            if (edit_type != "") save_grid();
            edit_type = "ACT";
            dataGridView1.Columns.Clear();
            dataGridView1.Rows.Clear();
            dataGridView1.Columns.Add("account", "Account");
            dataGridView1.Columns.Add("desc", "Description");
            dataGridView1.Columns.Add("major", "Major code");
            dataGridView1.Columns.Add("minor", "Minor code");
            dataGridView1.Columns.Add("stmt", "Statement");
            int size = Form1.db.accounts.Count;
            if (size > 0)
            {
                foreach (account data in Form1.db.accounts.Values)
                {
                    string[] row = {data.account_number.ToString() , data.description, data.major, data.minor, data.statement.ToUpper()};
                    dataGridView1.Rows.Add(row);
                }
            }
        }

        private void maintainper_Click(object sender, EventArgs e)
        {
            if (edit_type != "") save_grid();
            edit_type = "PER";
            dataGridView1.Columns.Clear();
            dataGridView1.Rows.Clear();
            dataGridView1.Columns.Add("year", "Year");
            dataGridView1.Columns.Add("period", "Period");
            dataGridView1.Columns.Add("ending", "Ending date");
            int size = Form1.db.periods.Count;
            if (size > 0)
            {
                foreach (financial_period data in Form1.db.periods.Values)
                {
                    string[] row = { data.year.ToString() , data.period.ToString() , data.ending.ToString(date_format) };
                    dataGridView1.Rows.Add(row);
                }
            }
        }

        private void maintainbatch_Click(object sender, EventArgs e)
        {
            if (edit_type != "") save_grid();
            edit_type = "BAT";
            dataGridView1.Columns.Clear();
            dataGridView1.Rows.Clear();
            dataGridView1.Columns.Add("journal", "Journal");
            dataGridView1.Columns.Add("date", "Date");
            dataGridView1.Columns.Add("time", "Time");
            dataGridView1.Columns.Add("year", "Year");
            dataGridView1.Columns.Add("period", "Period");
            dataGridView1.Columns.Add("desc", "Description");
            dataGridView1.Columns.Add("source", "Source");
            dataGridView1.Columns.Add("user", "User ID");
            dataGridView1.Columns.Add("type", "Type");
            int size = Form1.db.jheaders.Count;
            if (size > 0)
            {
                foreach (journal_header data in Form1.db.jheaders.Values)
                {
                    string[] row = { data.journal.ToString(), data.detail_date_time.ToString(date_format),
                                     data.detail_date_time.ToShortTimeString(), data.year.ToString(),
                                     data.period.ToString(), data.description, data.source, data.userid, data.type.ToUpper() };
                    dataGridView1.Rows.Add(row);
                }
            }
        }

        private void add_income_accounts() //retained earnings account on a balance sheet
                                           //  must include the totals from all the income statement accounts
        {
            foreach (account data in Form1.db.accounts.Values)
            {
                if (data.statement == "I")
                {
                    string key = string.Format("{0,20}{1,4:0000}{2,2:00}", data.account_number, print_year, print_period);
                    if (Form1.db.summaries.ContainsKey(key)) amount += Form1.db.summaries[key].total;
                    //add previous period totals to amount as well
                    for (int i = print_period - 1; i >= 0; i--)
                    {
                        string key2 = string.Format("{0,20}{1,4:0000}{2,2:00}", data.account_number, print_year, i);
                        if (Form1.db.summaries.ContainsKey(key2)) amount += Form1.db.summaries[key2].total;
                    }
                }
            }
        }

        private void closeyear_Click(object sender, EventArgs e)
        {
            if (edit_type != "") save_grid();
            dataGridView1.Columns.Clear();
            dataGridView1.Rows.Clear();
            int retained_account = 0;
            Int32.TryParse(reacct.Text, out retained_account);
            if (!Form1.db.accounts.ContainsKey(retained_account)) 
                MessageBox.Show("Cannot close since unknown retained account");
            else
            {
                askyear data = new askyear();
                data.ShowDialog();
                if (data.year > 0)
                {
                    db.save(); //save before processing
                    int count = 0, max_period = 13, last_account = 0;
                    string last_statement = "";
                    print_year = data.year;
                    int next_year = print_year + 1;
                    foreach (financial_period per in Form1.db.periods.Values)
                        if (per.year == print_year) max_period = per.period;
                    print_period = max_period;

                    bool found_retained = false;
                    double total = 0;
                    account master = new account();
                    List<string> keys = new List<string>(Form1.db.summaries.Keys);
                    foreach (string getkey in keys) //transfer year totals to next year opening balance
                    {
                        summary record = new summary();
                        record = Form1.db.summaries[getkey];
                        if (record.year == print_year)
                        {
                            if (record.account_number != last_account)
                            {
                                count++;
                                if (last_account > 0 && last_statement == "B") //put total into next year
                                {
                                    string key = string.Format("{0,20}{1,4:0000}{2,2:00}", last_account, next_year, 0);
                                    summary rec_next = new summary();
                                    rec_next.account_number = last_account;
                                    if (last_account == retained_account) found_retained = true;
                                    rec_next.year = next_year;
                                    rec_next.period = 0;
                                    rec_next.total = total;
                                    Form1.db.summaries[key] = rec_next;
                                }
                                master = new account();
                                if (Form1.db.accounts.ContainsKey(record.account_number)) master = Form1.db.accounts[record.account_number];
                                total = 0;
                            }
                            if (master.statement == "B") total += record.total;
                            last_statement = master.statement;
                            last_account = master.account_number;
                        }
                    }
                    //put in totals for last account
                    if (last_account > 0 && last_statement == "B")
                    {
                        count++;
                        string key = string.Format("{0,20}{1,4:0000}{2,2:00}", last_account, next_year, 0);
                        summary rec_next = new summary();
                        rec_next.account_number = last_account;
                        rec_next.year = next_year;
                        rec_next.period = 0;
                        rec_next.total = total;
                        Form1.db.summaries[key] = rec_next;
                    }
                    //now add up income amount to retained earnings account
                    amount = 0;
                    add_income_accounts(); //get total amount of all income accounts in year to close 
                    summary rec = new summary();
                    string rkey = string.Format("{0,20}{1,4:0000}{2,2:00}", retained_account, next_year, 0);
                    if (Form1.db.summaries.ContainsKey(rkey)) rec = Form1.db.summaries[rkey];
                    rec.account_number = retained_account;
                    rec.year = next_year;
                    rec.period = 0;
                    if (found_retained) rec.total += amount; //add to retained balance if funds found in current year
                    //  in order to move existing balances forward to next year
                    else rec.total = amount; //reset retained balance if no opening balances found in current year
                    Form1.db.summaries[rkey] = rec;
                    db.save(); //save after processing
                    MessageBox.Show("Year has been closed for " + count + " accounts");
                }
                else MessageBox.Show("Year End Cancelled");
            }
        }

        private void showtrial_Click(object sender, EventArgs e)
        {
            report_type = "TB";
            report();
        }

        private void showtrans_Click(object sender, EventArgs e)
        {
            report_type = "TR";
            report();
        }

        private void incomestmt_Click(object sender, EventArgs e)
        {
            report_type = "IN";
            report();
        }

        private void balancesheet_Click(object sender, EventArgs e)
        {
            report_type = "BL";
            report();
        }

        private void report()
        {
            if (edit_type != "") save_grid();
            askoptions data = new askoptions();
            data.ShowDialog();
            print_ytd = data.ytd;
            print_year = data.year;
            print_period = data.period;
            if (print_year > 0)
            {
                DialogResult print=printDialog1.ShowDialog();
                if (print == DialogResult.OK)
                {
                    printDocument1.PrinterSettings.PrinterName = printDialog1.PrinterSettings.PrinterName;
                    printPreviewDialog1.PrintPreviewControl.Zoom = 1.0;
                    printPreviewDialog1.Width = 900;
                    printPreviewDialog1.Document = printDocument1;
                    //reset reporting variables
                    column = row = lines_used = last_print_account = 0;
                    major_total = minor_total = final_total = tran_total = net_total = 0;
                    last_major = last_minor = report_data = print_amount = "";
                    last_print_trans_account = "";
                    heading_printed = fullpage = false;
                    print_header = true;
                    print_retained_account = 0;
                    Int32.TryParse(reacct.Text, out print_retained_account);
                    printPreviewDialog1.ShowDialog(); //calls PrintPage method of PrintDocument control
                }
                else MessageBox.Show("Report Cancelled");
            }
            else MessageBox.Show("Report Cancelled");
        }

        private string edit_amount(double amount)
        {
            string format;
            format=string.Format("{0,15:0.00;(0.00);0.00}", amount);
            if (amount < 0) format = " " + format; //move over bracketed negative amounts
            return format;
        }

        private void printDocument1_PrintPage(object sender, System.Drawing.Printing.PrintPageEventArgs e)
        {//this method is re-entered each time a page is filled up and so it must keep track of where it left off

            // Print report header 

            if (print_header)
            {
                e.Graphics.DrawString(companyname.Text, f, b, column, row);
                row = row + font_size + 2; //increase row and line spacing
                if (report_type == "TB") report_description = "Trial Balance";
                if (report_type == "TR") report_description = "Transactions";
                if (report_type == "IN") report_description = "Income Statement";
                if (report_type == "BL")
                {
                    report_description = "Balance Sheet";
                    print_ytd = "Y"; //balance sheet is always shown as year to date values in accounting
                }
                e.Graphics.DrawString(report_description, f, b, column, row);
                row = row + font_size + 2;
                if (print_ytd == "Y")
                {
                    report_description = "For year ending ";
                    if (report_type == "BL" || report_type == "TB")
                        report_description = "As at ";
                }
                else report_description = "For period ending ";
                printkey = string.Format("{0,4:0000}{1,2:00}", print_year, print_period);
                print_date = new DateTime();
                print_date = DateTime.Now;
                foundperiod = false;
                if (Form1.db.periods.ContainsKey(printkey))
                {
                    foundperiod = true;
                    print_date = Form1.db.periods[printkey].ending;
                    report_description = report_description + print_date.ToString(date_format);
                }
                else
                {
                    report_description += (print_year.ToString() + "/" + print_period.ToString());
                }
                e.Graphics.DrawString(report_description, f, b, column, row);
                row = row + 2 * font_size + 2; //jump down 2 lines before printing data
                print_header = false;
                lines_used = 4;
            }

            // Print report details

            foreach (account data in Form1.db.accounts.Values)
            {//check if printpage method is being reentered due to a full page (previously printed data is ignored)
                if (report_type == "TR" && data.account_number >= last_print_account //restart in middle of transactions
                    || report_type != "TR" && data.account_number > last_print_account) //restart at next account
                {
                    last_print_account = data.account_number;
                    printkey = string.Format("{0,20}{1,4:0000}{2,2:00}", data.account_number, print_year, print_period);
                    amount = 0;
                    if (Form1.db.summaries.ContainsKey(printkey)) amount = Form1.db.summaries[printkey].total;
                    if (print_ytd == "Y") //add previous period totals to amount as well
                    {
                        for (int i = print_period - 1; i >= 0; i--)
                        {
                            string key2 = string.Format("{0,20}{1,4:0000}{2,2:00}", data.account_number, print_year, i);
                            if (Form1.db.summaries.ContainsKey(key2)) amount += Form1.db.summaries[key2].total;
                        }
                    }
                    if (data.account_number == print_retained_account && report_type == "BL")
                    {
                        add_income_accounts(); //add all income activity into the balance sheet
                    }
                    final_total += amount;
                    print_amount = edit_amount(amount);
                    if (report_type == "TB")
                    {
                        if(foundperiod)
                            report_data += (string.Format("{0,-35}{1,15}", data.description, print_amount) + "\n");
                        else
                            report_data += (string.Format("{0,5} {1,-35}{2,15}",data.account_number, data.description, print_amount) + "\n");
                        lines_used++;
                    }
                    if (report_type == "IN" && data.statement == "I" || report_type == "BL" && data.statement == "B")
                    {
                        if (data.minor != last_minor && last_minor != "") //print minor total
                        {
                            report_data += (string.Format("{0,45}{1,15}", " ", edit_amount(minor_total)) + "\n");
                            lines_used++;
                            minor_total = 0;
                        }
                        if (data.major != last_major && last_major != "") //print major total
                        {
                            report_data += ("\n" + string.Format("{0,55}{1,15}", " ", edit_amount(major_total)) + "\n");
                            lines_used += 2;
                            major_total = 0;
                        }
                        if (last_major != data.major) //print major heading
                        {
                            last_major = data.major;
                            string title = "";
                            if (Form1.db.categories.ContainsKey(data.major)) title = Form1.db.categories[data.major].description;
                            if (title != "")
                            {
                                report_data += ("\n" + title + "\n\n");
                                lines_used += 3;
                            }
                        }
                        if (last_minor != data.minor) //print minor heading
                        {
                            last_minor = data.minor;
                            string title = "";
                            if (Form1.db.categories.ContainsKey(data.minor)) title = Form1.db.categories[data.minor].description;
                            if (title != "")
                            {
                                report_data += ("\t" + title + "\n");
                                lines_used++;
                            }
                        }
                        report_data += (string.Format("{0,-35}{1,15}", data.description, print_amount) + "\n");
                        lines_used++;
                        minor_total += amount;
                        major_total += amount;
                        net_total += amount;
                    }
                    if (report_type == "TR")
                    {
                        // get start of transaction details in current period and print them
                        if (print_ytd == "Y") //year to date data must start at first available period
                        {
                            bool looking = true;
                            int look_period = 0;
                            while (looking && look_period <= print_period)
                            {
                                printkey = string.Format("{0,20}{1,4:0000}{2,2:00}", 
                                                                            data.account_number, print_year, look_period);
                                start_key = string.Format("{0,26}{1,32}", printkey, " ");
                                start = Form1.db.details.IndexOfKey(start_key);
                                if (start >= 0) looking = false; //found first period
                                else look_period++;
                            }
                        }
                        start_key = string.Format("{0,26}{1,32}", printkey, " ");
                        //check if printpage method has been re-entered in middle of transactions due to a full page
                        if (last_print_trans_account != "") start_key = last_print_trans_account; //find last transaction
                        start = Form1.db.details.IndexOfKey(start_key);
                        if (last_print_trans_account != "") start++; //skip to next transaction
                        //print transactions
                        if (start >= 0)
                        {
                            detail record = new detail();
                            for (int i = start; i < Form1.db.details.Count; i++)
                            {
                                string detail_key = Form1.db.details.Keys[i];
                                last_print_trans_account = detail_key;
                                record = Form1.db.details[detail_key];
                                if (record.account_number != data.account_number) break;
                                if (record.year != print_year) break;
                                if (record.period > print_period) break;
                                tran_total += record.amount;
                                if (record.amount != 0)
                                {
                                    if (!heading_printed)
                                    {
                                        report_data += (string.Format("{0,-35}", data.description) + "\n");
                                        heading_printed = true;
                                        lines_used++;
                                    }
                                    //adjust output lengths since string.format expands for longer data
                                    string line_description = record.description;
                                    if (line_description.Length > 25) line_description = line_description.Substring(0, 25);
                                    string line_source = record.source;
                                    if (line_source.Length > 10) line_source = line_source.Substring(0, 10);
                                    string line_userid = record.userid;
                                    if (line_userid.Length > 9) line_userid = line_userid.Substring(0, 9);
                                    report_data += (string.Format("{0,10}{1,11} {2,-25}{3,-10}{4,9}{5,15}",
                                    record.journal, record.detail_date_time.ToString(date_format),
                                    line_description, line_source, line_userid,
                                    edit_amount(record.amount)) + "\n");
                                    lines_used++;
                                    if (lines_used > 50)
                                    {
                                        fullpage = true;
                                        break;
                                    }
                                }
                            }
                            if(!fullpage) last_print_trans_account = "";
                        }
                        else last_print_trans_account = "";
                        if (!fullpage)
                        {
                            if (heading_printed)
                            {
                                report_data += (string.Format("{0,66}{1,15}", "TOTAL", edit_amount(tran_total)) + "\n");
                                last_print_account++;
                                lines_used++;
                            }
                            heading_printed = false;
                            tran_total = 0;
                        }
                    }
                    if (lines_used > 50 || fullpage) //check if current loop filled page or if inner loop filled page
                    {
                        fullpage = true;
                        break;
                    }
                }
            }
            if (!fullpage) //print final totals if page is not full
            {
                if (report_type == "TB")
                {
                    report_data += ("\n" + string.Format("{0,-35}{1,15}", "TOTAL", edit_amount(final_total)) + "\n");
                    lines_used += 2;
                }
                if (report_type == "TR" && heading_printed)
                {
                    report_data += (string.Format("{0,66}{1,15}", "TOTAL", edit_amount(tran_total)) + "\n");
                    lines_used++;
                }
                if (report_type == "IN" || report_type == "BL")
                {
                    report_data += (string.Format("{0,45}{1,15}", " ", edit_amount(minor_total)) + "\n");
                    report_data += ("\n" + string.Format("{0,55}{1,15}", " ", edit_amount(major_total)) + "\n");
                    report_data += ("\n" + string.Format("{0,55}{1,15}", "NET", edit_amount(net_total)) + "\n");
                    lines_used += 5;
                }
            }
            e.Graphics.DrawString(report_data, f, b, column, row); //show current page of data
            report_data = "";
            if (fullpage)
            {
                e.HasMorePages = true;
                lines_used = row = 0;
                fullpage = false;
            }
            else e.HasMorePages = false;
        }

        private void processbatch_Click(object sender, EventArgs e)
        {
            if (edit_type != "") save_grid();
            dataGridView1.Columns.Clear();
            dataGridView1.Rows.Clear();
            //check for errors in batches first eg. not balanced, zero year, bad type, future batch, unknown account
            string message = "";
            ArrayList good = new ArrayList();
            int years=current_year, per=current_period;
            foreach (journal_header head in Form1.db.jheaders.Values)
            {
                string id = string.Format("{0,10}", head.journal);
                if (head.type != "N" && head.type != "A" && head.type != "R")
                    message += (id + " unknown type;");
                else
                    if (head.year == 0 || head.period==0)
                        message += (id + " unknown year/period;");
                    else
                        if (head.type == "R" && (head.year != years || head.period != per))
                            message += (id + " waiting to process in period " + head.period + ";");
                        else
                        {
                            double total = 0;
                            bool good_account = true;
                            foreach (journal_detail detail in Form1.db.jdetails.Values)
                                if (detail.journal == head.journal)
                                {
                                    total += detail.amount;
                                    if (!Form1.db.accounts.ContainsKey(detail.account_number))
                                    {
                                        good_account = false;
                                        message += (id + " " + detail.account_number + " unknown account;");
                                    }
                                }
                            if (total != 0)
                                message += (id + " out of balance by " 
                                    + string.Format("{0,10:0.00;(0.00);0.00}",total) + ";");
                            else
                                if(good_account)
                                {
                                    message += (id + " IS OK TO POST.;");
                                    good.Add(head.journal);
                                }
                        }
            }
            message += " ";
            askok ask = new askok(message);
            ask.ShowDialog();

            if (ask.ok)
            {
                db.save(); //save before processing
                int count = 0;
                
                foreach (int batch in good)
                {
                    journal_header head = new journal_header();
                    head = Form1.db.jheaders[batch];
                    int next_year = head.year, next_period = head.period, max_period = 13;
                    //find next processing period for accruals and recurring batches
                    foreach (financial_period data in Form1.db.periods.Values)
                        if (data.year == head.year) max_period = data.period;
                    next_period++;
                    if (next_period > max_period)
                    {
                        next_period = 1;
                        next_year++;
                    }
                    List<string> keys = new List<string>(Form1.db.jdetails.Keys);
                    foreach (string getkey in keys)
                    {
                        journal_detail det = new journal_detail();
                        det = Form1.db.jdetails[getkey];
                        if (det.journal == batch)
                        {
                            string summary_key = string.Format("{0,20}{1,4:0000}{2,2:00}",
                                                          det.account_number, head.year, head.period);
                            string next_summary_key = string.Format("{0,20}{1,4:0000}{2,2:00}",
                                                          det.account_number, next_year, next_period);
                            //start keys are used to setup empty records so that SortedList indexing works
                            //  since lists do not have the LIKE search that can be found in databases
                            string start_summary_key = string.Format("{0,20}{1,4:0000}{2,2:00}",
                                                          det.account_number, head.year, 0);
                            string next_start_summary_key = string.Format("{0,20}{1,4:0000}{2,2:00}",
                                                          det.account_number, next_year, 0);
                            string detail_key = string.Format("{0,26}{1,8}{2,4}{3,20}",
                                                    summary_key, head.detail_date_time.ToString("yyyyMMdd"),
                                                    head.detail_date_time.ToString("hhmm"), head.journal);
                            string next_detail_key = string.Format("{0,26}{1,8}{2,4}{3,20}",
                                                    next_summary_key, head.detail_date_time.ToString("yyyyMMdd"),
                                                    head.detail_date_time.ToString("hhmm"), head.journal);
                            string start_detail_key = string.Format("{0,26}{1,32}", summary_key, " ");
                            string next_start_detail_key = string.Format("{0,26}{1,32}", next_summary_key, " ");
                            //first add to detail activity records
                            detail record = new detail();
                            record.account_number = det.account_number;
                            record.year = head.year;
                            record.period = head.period;
                            record.journal = head.journal;
                            record.detail_date_time = head.detail_date_time;
                            record.description = head.description;
                            record.source = head.source;
                            record.userid = head.userid;
                            record.amount = det.amount;
                            Form1.db.details[detail_key] = record;
                            if (!Form1.db.details.ContainsKey(start_detail_key))
                            {
                                record = new detail();
                                record.account_number = det.account_number;
                                record.year = head.year;
                                record.period = head.period;
                                Form1.db.details[start_detail_key] = record;
                            }
                            if (head.type == "A") //accruals must reverse in next period
                            {
                                record = new detail();
                                record.account_number = det.account_number;
                                record.year = next_year;
                                record.period = next_period;
                                record.journal = head.journal;
                                record.detail_date_time = head.detail_date_time;
                                record.description = head.description;
                                record.source = head.source;
                                record.userid = head.userid;
                                record.amount = -1 * det.amount;
                                Form1.db.details[next_detail_key] = record;
                                if (!Form1.db.details.ContainsKey(next_start_detail_key))
                                {
                                    record = new detail();
                                    record.account_number = det.account_number;
                                    record.year = next_year;
                                    record.period = next_period;
                                    Form1.db.details[next_start_detail_key] = record;
                                }
                            }
                            if (head.type == "A" || head.type == "N") Form1.db.jdetails.Remove(string.Format("{0,20}{1,20}",
                                                                            head.journal, det.account_number));
                            //now update summary totals
                            summary summary_record = new summary();
                            if (Form1.db.summaries.ContainsKey(summary_key)) summary_record = Form1.db.summaries[summary_key];
                            summary_record.account_number = det.account_number;
                            summary_record.year = head.year;
                            summary_record.period = head.period;
                            summary_record.total += det.amount;
                            Form1.db.summaries[summary_key] = summary_record;
                            if (!Form1.db.summaries.ContainsKey(start_summary_key))
                            {
                                summary_record = new summary();
                                summary_record.account_number = det.account_number;
                                summary_record.year = head.year;
                                Form1.db.summaries[start_summary_key] = summary_record;
                            }
                            if (head.type == "A") //accruals must add opposite amounts to next period
                            {
                                summary_record = new summary();
                                if (Form1.db.summaries.ContainsKey(next_summary_key)) summary_record = Form1.db.summaries[next_summary_key];
                                summary_record.account_number = det.account_number;
                                summary_record.year = next_year;
                                summary_record.period = next_period;
                                summary_record.total -= det.amount;
                                Form1.db.summaries[next_summary_key] = summary_record;
                                if (!Form1.db.summaries.ContainsKey(next_start_summary_key))
                                {
                                    summary_record = new summary();
                                    summary_record.account_number = det.account_number;
                                    summary_record.year = next_year;
                                    Form1.db.summaries[next_start_summary_key] = summary_record;
                                }
                            }
                        }
                    }
                    if (head.type == "A" || head.type == "N") Form1.db.jheaders.Remove(head.journal);
                    if (head.type == "R") //update to next period for recurring journals
                    {
                        head.year = next_year;
                        head.period = next_period;
                        Form1.db.jheaders[head.journal] = head;
                    }
                    count++;
                }
                db.save(); //save after processing
                ejournal.Text = "";
                last_journal = 0;
                type.Text = "";
                year.Text = "";
                period.Text = "";
                source.Text = "";
                description.Text = "";
                dataGridView1.Columns.Clear();
                dataGridView1.Rows.Clear();
                get_highest_journal();
                MessageBox.Show(count + " batches processed.");
            }
        }

        private void company_TextChanged(object sender, EventArgs e)
        {
            reacct_TextChanged(sender, e); //just use storage code for other control field        
        }

        private void reacct_TextChanged(object sender, EventArgs e)
        {
            if (loaded)
            {
                control data = new control();
                data.company = companyname.Text;
                int account = 0;
                Int32.TryParse(reacct.Text, out account);
                data.reacct = account;
                Form1.db.controls[1] = data; //control data is always stored in position one
            }
        }

        private void show_details(int journal)
        {
            if (edit_type != "") save_grid();
            edit_type = "DET";
            dataGridView1.Columns.Clear();
            dataGridView1.Rows.Clear();
            dataGridView1.Columns.Add("account", "Account");
            dataGridView1.Columns.Add("amount", "Amount");
            int size = Form1.db.jdetails.Count;
            if (size > 0)
            {
                foreach (journal_detail data in Form1.db.jdetails.Values)
                {
                    if (data.journal == journal)
                    {
                        string[] row = { data.account_number.ToString(), data.amount.ToString() };
                        dataGridView1.Rows.Add(row);
                    }
                }
            }
        }

        private void ejournal_TextChanged(object sender, EventArgs e) //setup new journal or display existing journal
        {
            if (edit_type != "") save_grid();
            if (loaded)
            {
                int journal = 0;
                Int32.TryParse(ejournal.Text, out journal);
                if (Form1.db.jheaders.ContainsKey(journal))
                {
                    header = new journal_header();
                    header = Form1.db.jheaders[journal];
                    description.Text = header.description;
                    source.Text = header.source;
                    year.Text = header.year.ToString();
                    period.Text = header.period.ToString();
                    type.Text = header.type;
                }
                else
                {
                    description.Text = "";
                    source.Text = "";
                    type.Text = "N";
                    year.Text = current_year.ToString();
                    period.Text = current_period.ToString();
                }
                show_details(journal);
                last_journal = journal;
            }
        }

        private void ejournal_Click(object sender, EventArgs e) //load data for displayed journal
        {
            if (edit_type != "") save_grid();
            if (loaded)
            {
                int journal_number = 0;
                Int32.TryParse(ejournal.Text, out journal_number);
                if (journal_number != 0)
                {
                    header = new journal_header();
                    if (Form1.db.jheaders.ContainsKey(journal_number)) header = Form1.db.jheaders[journal_number];
                    header.detail_date_time = DateTime.Now;
                    header.userid = current_user;
                    header.journal = journal_number;
                    year.Text = header.year.ToString();
                    if (header.year == 0) year.Text = current_year.ToString();
                    period.Text = header.period.ToString();
                    if (header.period == 0) period.Text = current_period.ToString();
                    description.Text = header.description;
                    source.Text = header.source;
                    type.Text = header.type;
                    show_details(journal_number);
                    last_journal = journal_number;
                }
            }
        }

        private void journal_SelectedIndexChanged(object sender, EventArgs e) //user selected data from dropdown journal list
        {
            if (edit_type != "") save_grid();
            if (loaded && process)
            {
                int journal_number = 0;
                Int32.TryParse(journal.Text, out journal_number);
                header = new journal_header();
                header.journal = journal_number;
                if (Form1.db.jheaders.ContainsKey(journal_number))
                {
                    header = Form1.db.jheaders[journal_number];
                    ejournal.Text = journal.Text;
                    description.Text = header.description;
                    source.Text = header.source;
                    year.Text = header.year.ToString();
                    period.Text = header.period.ToString();
                    if (header.year == 0) year.Text = current_year.ToString();
                    if (header.period == 0) period.Text = current_period.ToString();
                    type.Text = header.type;
                    show_details(journal_number);
                    last_journal = journal_number;
                }
            }
        }

        private void journal_Click(object sender, EventArgs e) //refresh list of journals
        {
            loadjournals();
        }

    }
    [Serializable()]
    public class control
    {
        public string company { get; set; }
        public int reacct { get; set; }
    }
    [Serializable()]
    public class category
    {
        public string code { get; set; }
        public string description { get; set; }
    }
    [Serializable()]
    public class financial_period
    {
        public int year { get; set; }
        public int period { get; set; }
        public DateTime ending { get; set; }
    }
    [Serializable()]
    public class account
    {
        public int account_number { get; set; }
        public string description { get; set; }
        public string major { get; set; }
        public string minor { get; set; }
        public string statement { get; set; }
    }
    [Serializable()]
    public class summary
    {
        public int account_number { get; set; }
        public int year { get; set; }
        public int period { get; set; }
        public double total { get; set; }
    }
    [Serializable()]
    public class detail
    {
        public int account_number { get; set; }
        public int year { get; set; }
        public int period { get; set; }
        public int journal { get; set; }
        public DateTime detail_date_time { get; set; }
        public string description { get; set; }
        public string source { get; set; }
        public double amount { get; set; }
        public string userid { get; set; }
    }
    [Serializable()]
    public class journal_detail
    {
        public int journal { get; set; }
        public int account_number { get; set; }
        public double amount { get; set; }
    }
    [Serializable()]
    public class journal_header
    {
        public int journal { get; set; }
        public DateTime detail_date_time { get; set; }
        public int year { get; set; }
        public int period { get; set; }
        public string description { get; set; }
        public string source { get; set; }
        public string userid { get; set; }
        public string type { get; set; }
    }
    [Serializable()]
    public class database //original class using sortedlists stored in a compressed binary format
    {// this was changed to new databaseglerp class
     // to allow XML file creation that is easy to change for future year demo data (just do scan/replace years)
        public SortedList<int, control> controls = new SortedList<int, control>();
        public SortedList<string, category> categories = new SortedList<string, category>();
        public SortedList<int, account> accounts = new SortedList<int, account>();
        public SortedList<string, financial_period> periods = new SortedList<string, financial_period>();
        public SortedList<string, summary> summaries = new SortedList<string, summary>();
        public SortedList<string, detail> details = new SortedList<string, detail>();
        public SortedList<int, journal_header> jheaders = new SortedList<int, journal_header>();
        public SortedList<string, journal_detail> jdetails = new SortedList<string, journal_detail>();
        //Note: For multi-key data structures, the keys are formatted together using String.Format

        public void save()
        {
            BinaryFormatter b = new BinaryFormatter();
            FileStream f = new FileStream(@"GLERP.DAT", FileMode.Create);
            b.Serialize(f, this);
            f.Close();
        }
        public database load()
        {
            BinaryFormatter b = new BinaryFormatter();
            using (FileStream f = new FileStream(@"GLERP.DAT", FileMode.Open))
            {
                return (database)b.Deserialize(f);
            }
        }
    }
    [Serializable()]
    public class databaseglerp
    {//note that microsoft XML cannot store sortedlists and so they have been marked as XMLIgnore to keep out of file
     //and replaced here by matching lists in the class for their keys and data
     //(this used to be class database with sortedlists that were stored in a binary compressed format)
     //(to now use XML, keys and data must be moved back and forth between lists and sortedlists)
     //(sortedlists are still used in program as they can be manipulated like database tables with keys)
        [XmlIgnore]
        public SortedList<int, control> controls = new SortedList<int, control>();
        public List<int> controlskeys = new List<int>();
        public List<control> controlsdata = new List<control>();
        [XmlIgnore]
        public SortedList<string, category> categories = new SortedList<string, category>();
        public List<string> categorieskeys = new List<string>();
        public List<category> categoriesdata = new List<category>();
        [XmlIgnore]
        public SortedList<int, account> accounts = new SortedList<int, account>();
        public List<int> accountskeys = new List<int>();
        public List<account> accountsdata = new List<account>();
        [XmlIgnore]
        public SortedList<string, financial_period> periods = new SortedList<string, financial_period>();
        public List<string> periodskeys = new List<string>();
        public List<financial_period> periodsdata = new List<financial_period>();
        [XmlIgnore]
        public SortedList<string, summary> summaries = new SortedList<string, summary>();
        public List<string> summarieskeys = new List<string>();
        public List<summary> summariesdata = new List<summary>();
        [XmlIgnore]
        public SortedList<string, detail> details = new SortedList<string, detail>();
        public List<string> detailskeys = new List<string>();
        public List<detail> detailsdata = new List<detail>();
        [XmlIgnore]
        public SortedList<int, journal_header> jheaders = new SortedList<int, journal_header>();
        public List<int> jheaderskeys = new List<int>();
        public List<journal_header> jheadersdata = new List<journal_header>();
        [XmlIgnore]
        public SortedList<string, journal_detail> jdetails = new SortedList<string, journal_detail>();
        public List<string> jdetailskeys = new List<string>();
        public List<journal_detail> jdetailsdata = new List<journal_detail>();
        //Note: For multi-key data structures, the keys are formatted together using String.Format

        public void save()
        {
            //for each sortedlist need this special code to move sortedlists to lists to make into XML
            int x1;
            //--------------------------------------
            controlskeys = new List<int>();
            controlsdata = new List<control>();
            for(x1=0;x1<Form1.db.controls.Count;x1++)
            {
                controlskeys.Add(Form1.db.controls.Keys[x1]);
                controlsdata.Add(Form1.db.controls.Values[x1]);
            }
            //--------------------------------------
            categorieskeys = new List<string>();
            categoriesdata = new List<category>();
            for (x1 = 0; x1 < Form1.db.categories.Count; x1++)
            {
                categorieskeys.Add(Form1.db.categories.Keys[x1]);
                categoriesdata.Add(Form1.db.categories.Values[x1]);
            }
            //--------------------------------------
            accountskeys = new List<int>();
            accountsdata = new List<account>();
            for (x1 = 0; x1 < Form1.db.accounts.Count; x1++)
            {
                accountskeys.Add(Form1.db.accounts.Keys[x1]);
                accountsdata.Add(Form1.db.accounts.Values[x1]);
            }
            //--------------------------------------
            periodskeys = new List<string>();
            periodsdata = new List<financial_period>();
            for (x1 = 0; x1 < Form1.db.periods.Count; x1++)
            {
                periodskeys.Add(Form1.db.periods.Keys[x1]);
                periodsdata.Add(Form1.db.periods.Values[x1]);
            }
            //--------------------------------------
            summarieskeys = new List<string>();
            summariesdata = new List<summary>();
            for (x1 = 0; x1 < Form1.db.summaries.Count; x1++)
            {
                summarieskeys.Add(Form1.db.summaries.Keys[x1]);
                summariesdata.Add(Form1.db.summaries.Values[x1]);
            }
            //--------------------------------------
            detailskeys = new List<string>();
            detailsdata = new List<detail>();
            for (x1 = 0; x1 < Form1.db.details.Count; x1++)
            {
                detailskeys.Add(Form1.db.details.Keys[x1]);
                detailsdata.Add(Form1.db.details.Values[x1]);
            }
            //--------------------------------------
            jheaderskeys = new List<int>();
            jheadersdata = new List<journal_header>();
            for (x1 = 0; x1 < Form1.db.jheaders.Count; x1++)
            {
                jheaderskeys.Add(Form1.db.jheaders.Keys[x1]);
                jheadersdata.Add(Form1.db.jheaders.Values[x1]);
            }
            //--------------------------------------
            jdetailskeys = new List<string>();
            jdetailsdata = new List<journal_detail>();
            for (x1 = 0; x1 < Form1.db.jdetails.Count; x1++)
            {
                jdetailskeys.Add(Form1.db.jdetails.Keys[x1]);
                jdetailsdata.Add(Form1.db.jdetails.Values[x1]);
            }
            //BinaryFormatter b = new BinaryFormatter();
            XmlSerializer b2 = new XmlSerializer(typeof(databaseglerp));
            FileStream f = new FileStream(@"glerp.xml", FileMode.Create);
            b2.Serialize(f, this);
            f.Close();
        }
        public void load()
        {
            //BinaryFormatter b = new BinaryFormatter();
            XmlSerializer b2 = new XmlSerializer(typeof(databaseglerp));
            using (FileStream f = new FileStream(@"glerp.xml", FileMode.Open))
            {
                Form1.db=(databaseglerp)b2.Deserialize(f);
            }
            //for each sortedlist need this special code to move lists to sortedlists in Form1 class
            int x1;
            //----------------------------------
            Form1.db.controls = new SortedList<int, control>();
            for (x1 = 0; x1 < Form1.db.controlskeys.Count; x1++)
                Form1.db.controls.Add(Form1.db.controlskeys[x1], Form1.db.controlsdata[x1]);
            //----------------------------------
            Form1.db.categories = new SortedList<string, category>();
            for (x1 = 0; x1 < Form1.db.categorieskeys.Count; x1++)
                Form1.db.categories.Add(Form1.db.categorieskeys[x1], Form1.db.categoriesdata[x1]);
            //----------------------------------
            Form1.db.accounts = new SortedList<int, account>();
            for (x1 = 0; x1 < Form1.db.accountskeys.Count; x1++)
                Form1.db.accounts.Add(Form1.db.accountskeys[x1], Form1.db.accountsdata[x1]);
            //----------------------------------
            Form1.db.periods = new SortedList<string, financial_period>();
            for (x1 = 0; x1 < Form1.db.periodskeys.Count; x1++)
                Form1.db.periods.Add(Form1.db.periodskeys[x1], Form1.db.periodsdata[x1]);
            //----------------------------------
            Form1.db.summaries = new SortedList<string, summary>();
            for (x1 = 0; x1 < Form1.db.summarieskeys.Count; x1++)
                Form1.db.summaries.Add(Form1.db.summarieskeys[x1], Form1.db.summariesdata[x1]);
            //----------------------------------
            Form1.db.details = new SortedList<string, detail>();
            for (x1 = 0; x1 < Form1.db.detailskeys.Count; x1++)
                Form1.db.details.Add(Form1.db.detailskeys[x1], Form1.db.detailsdata[x1]);
            //----------------------------------
            Form1.db.jheaders = new SortedList<int, journal_header>();
            for (x1 = 0; x1 < Form1.db.jheaderskeys.Count; x1++)
                Form1.db.jheaders.Add(Form1.db.jheaderskeys[x1], Form1.db.jheadersdata[x1]);
            //----------------------------------
            Form1.db.jdetails = new SortedList<string, journal_detail>();
            for (x1 = 0; x1 < Form1.db.jdetailskeys.Count; x1++)
                Form1.db.jdetails.Add(Form1.db.jdetailskeys[x1], Form1.db.jdetailsdata[x1]);
        }
    }

}
